<?php
// Run with: php aws_launch_import.php

define('CLI_SCRIPT', true);
require(__DIR__ . '/config.php');
require_once($CFG->dirroot . '/course/lib.php');
require_once($CFG->dirroot . '/mod/lti/lib.php');

global $DB;

// 1. Get AWS toolid from mdl_lti_types
$toolid = $DB->get_field('lti_types', 'id', ['name' => 'AWS'], IGNORE_MULTIPLE);
if (!$toolid) {
    cli_error("AWS LTI tool not found. Please check mdl_lti_types table.");
}

// 2. Get all courses under AWS category
$categoryid = $DB->get_field('course_categories', 'id', ['name' => 'AWS']);
if (!$categoryid) {
    cli_error("Category 'AWS' not found.");
}

$courses = $DB->get_records('course', ['category' => $categoryid]);

foreach ($courses as $course) {
    echo "Processing course: {$course->fullname} (ID {$course->id})\n";

    // 3. Get launchurl from custom field
    $sql = "SELECT d.value
              FROM {customfield_data} d
              JOIN {customfield_field} f ON f.id = d.fieldid
             WHERE d.instanceid = ? AND f.shortname = ?";
    $launchurl = $DB->get_field_sql($sql, [$course->id, 'launch_url']);

    if (empty($launchurl)) {
        echo "  ⚠ No launch URL found, skipping.\n";
        continue;
    }

    // 4. Check if activity already exists
    $exists = $DB->record_exists('lti', ['course' => $course->id, 'toolurl' => $launchurl]);
    if ($exists) {
        echo "  ✔ LTI activity already exists, skipping.\n";
        continue;
    }

    // 5. Prepare new LTI activity
    $lti = new stdClass();
    $lti->course      = $course->id;
    $lti->typeid      = $toolid;
    $lti->toolurl     = $launchurl;
    $lti->name        = "Launch AWS – {$course->fullname}";
    $lti->intro       = "Access AWS course content.";
    $lti->introformat = FORMAT_HTML;
    $lti->timecreated = time();
    $lti->timemodified= time();

    // Create LTI instance
    $instanceid = lti_add_instance($lti, null);

    if (!$instanceid) {
        echo "  ❌ Failed to create LTI activity\n";
        continue;
    }

    // Get module id for LTI
    $module = $DB->get_record('modules', ['name' => 'lti'], '*', MUST_EXIST);

    // Put it in section 0 (General)
    $section = $DB->get_record('course_sections', [
        'course' => $course->id,
        'section' => 0
    ], '*', MUST_EXIST);

    // Create course module entry
    $cm = new stdClass();
    $cm->course       = $course->id;
    $cm->module       = $module->id;
    $cm->instance     = $instanceid;
    $cm->section      = $section->id;
    $cm->visible      = 1;
    $cm->visibleoncoursepage = 1;
    $cm->added        = time();
    $cm->showdescription = 0;

    $cmid = $DB->insert_record('course_modules', $cm);

    // Add it into section sequence
    if ($section->sequence) {
        $section->sequence .= ',' . $cmid;
    } else {
        $section->sequence = $cmid;
    }
    $DB->update_record('course_sections', $section);

    // Rebuild course cache
    rebuild_course_cache($course->id, true);

    echo "  ✅ Created LTI activity and added to General section (cmid $cmid)\n";
}

echo "Done.\n";
