<?php
require_once(__DIR__ . '/../../config.php');
require_login();

global $DB, $USER, $OUTPUT, $PAGE;

// Set up page
$PAGE->set_url(new moodle_url('/local/custompage/course_req.php'));
$PAGE->set_context(context_system::instance());
$PAGE->set_title('Course Enrol Requests');
$PAGE->set_heading('Available Courses');

// Include DataTables
echo html_writer::tag('link', '', [
    'rel' => 'stylesheet',
    'type' => 'text/css',
    'href' => 'https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css'
]);
echo html_writer::tag('script', '', [
    'src' => 'https://code.jquery.com/jquery-3.6.0.min.js'
]);
echo html_writer::tag('script', '', [
    'src' => 'https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js'
]);

// Handle request button click
if (optional_param('request', 0, PARAM_INT)) {
    $courseid = required_param('courseid', PARAM_INT);

    $exists = $DB->record_exists('course_enrol_requests', [
        'courseid' => $courseid,
        'userid' => $USER->id
    ]);

    if (!$exists) {
        $record = new stdClass();
        $record->courseid = $courseid;
        $record->userid = $USER->id;
        $record->requested_at = time();
        $record->status = 0;
        $DB->insert_record('course_enrol_requests', $record);
    }
    redirect(new moodle_url('/local/custompage/course_req.php'));
}

// ✅ Fetch only courses NOT enrolled by user
$sql = "
    SELECT c.id, c.fullname, c.summary
    FROM {course} c
    WHERE c.id != 1
      AND c.id NOT IN (
          SELECT e.courseid
          FROM {enrol} e
          JOIN {user_enrolments} ue ON ue.enrolid = e.id
          WHERE ue.userid = :userid
      )
    ORDER BY c.fullname ASC
";
$courses = $DB->get_records_sql($sql, ['userid' => $USER->id]);

echo $OUTPUT->header();
echo html_writer::tag('h3', 'Available Courses', ['class' => 'mb-3']);

// Table structure
$table = new html_table();
$table->id = 'coursereq';
$table->head = ['Course Name', 'Content Type', 'Duration', 'Action'];

// Function to format duration
function format_duration($seconds) {
    if (empty($seconds) || !is_numeric($seconds)) return 'N/A';
    $h = floor($seconds / 3600);
    $m = floor(($seconds % 3600) / 60);
    $s = $seconds % 60;
    return sprintf("%dh %dm %ds", $h, $m, $s);
}

// Populate table
foreach ($courses as $course) {
    // Check if requested
    $requested = $DB->record_exists('course_enrol_requests', [
        'courseid' => $course->id,
        'userid' => $USER->id
    ]);

    // Fetch duration and content type
    $duration_value = $DB->get_field_sql("
        SELECT d.value
        FROM {customfield_data} d
        JOIN {customfield_field} f ON f.id = d.fieldid
        WHERE d.instanceid = :courseid AND f.shortname = 'course_duration_value'
    ", ['courseid' => $course->id]);

    $duration_unit = $DB->get_field_sql("
        SELECT d.value
        FROM {customfield_data} d
        JOIN {customfield_field} f ON f.id = d.fieldid
        WHERE d.instanceid = :courseid AND f.shortname = 'duration'
    ", ['courseid' => $course->id]);

    $total_seconds = 0;
    if (!empty($duration_value) && is_numeric($duration_value)) {
        switch (strtoupper($duration_unit)) {
            case 'HOURS': $total_seconds = $duration_value*3600; break;
            case 'MINUTES': $total_seconds = $duration_value*60; break;
            case 'SECONDS': $total_seconds = $duration_value; break;
            default: $total_seconds = $duration_value;
        }
    }
    $duration = format_duration($total_seconds);

    $contenttype = $DB->get_field_sql("
        SELECT d.value
        FROM {customfield_data} d
        JOIN {customfield_field} f ON f.id = d.fieldid
        WHERE d.instanceid = :courseid AND f.shortname = 'type'
    ", ['courseid' => $course->id]);
    $contenttype = $contenttype ?: 'N/A';

    $button = $requested
        ? html_writer::tag('button', 'Requested', ['class'=>'btn btn-secondary btn-sm', 'disabled'=>'disabled'])
        : html_writer::link(
            new moodle_url('/local/custompage/course_req.php', ['request'=>1,'courseid'=>$course->id]),
            'Request to Enrol',
            ['class'=>'btn btn-primary btn-sm']
        );

    $table->data[] = [
        format_string($course->fullname),
        format_string($contenttype),
        $duration,
        $button
    ];
}

echo html_writer::table($table);
?>

<!-- Table styling -->
<style>
#coursereq td a.btn,#coursereq td button.btn {white-space:nowrap !important;display:inline-block;max-width:none;padding:6px 12px;font-size:13px;}
#coursereq td:last-child {text-align:center;width:160px;}
</style>

<!-- DataTables -->
<script>
jQuery(document).ready(function($){
    $('#coursereq').DataTable({
        "pageLength": 10,
        "lengthMenu": [[10,25,50,100],[10,25,50,100]],
        "order": [],
        "language": {
            "search":"Search:",
            "lengthMenu":"Show _MENU_ entries",
            "info":"Showing _START_ to _END_ of _TOTAL_ courses",
            "paginate":{"first":"First","last":"Last","next":"Next","previous":"Previous"}
        }
    });
});
</script>

<?php
echo $OUTPUT->footer();
?>
