<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

define('NO_DEBUG_DISPLAY', true);
define('NO_MOODLE_COOKIES', true);

require_once(__DIR__ . '/../../config.php');
require_once($CFG->dirroot . '/mod/lti/locallib.php');

$response = new \mod_lti\local\ltiservice\response();
$isget = $response->get_request_method() === mod_lti\local\ltiservice\resource_base::HTTP_GET;
$isdelete = $response->get_request_method() === mod_lti\local\ltiservice\resource_base::HTTP_DELETE;

if ($isget) {
    $response->set_accept(isset($_SERVER['HTTP_ACCEPT']) ? $_SERVER['HTTP_ACCEPT'] : '');
} else {
    $response->set_content_type(isset($_SERVER['CONTENT_TYPE']) ? explode(';', $_SERVER['CONTENT_TYPE'], 2)[0] : '');
}

$ok = false;
$path = isset($_SERVER['PATH_INFO']) ? $_SERVER['PATH_INFO'] : '';

// === Unified Debug Log ===
$logfile = '/var/www/moodle/lti-debug.log';
$headers = "";
if (function_exists('getallheaders')) {
    foreach (getallheaders() as $name => $value) {
        $headers .= $name . ": " . $value . "\n";
    }
}
$body = file_get_contents('php://input');
file_put_contents(
    $logfile,
    "=== " . date('Y-m-d H:i:s') . " " . $_SERVER['REQUEST_METHOD'] . " " . $_SERVER['REQUEST_URI'] . " ===\n" .
    $headers . "\n" .
    $body . "\n\n",
    FILE_APPEND
);

$accept = $response->get_accept();
$contenttype = $response->get_content_type();

$services = lti_get_services();
foreach ($services as $service) {
    $resources = $service->get_resources();
    foreach ($resources as $resource) {
        if (($isget && !empty($accept) && (strpos($accept, '*/*') === false) &&
             !in_array($accept, $resource->get_formats())) ||
            ((!$isget && !$isdelete) && !in_array($contenttype, $resource->get_formats()))) {
            continue;
        }
        $template = $resource->get_template();
        $template = preg_replace('/{config_type}/', '(toolproxy|tool)', $template);
        $template = preg_replace('/\{[a-zA-Z_]+\}/', '[^/]+', $template);
        $template = preg_replace('/\(([0-9a-zA-Z_\-,\/]+)\)/', '(\\1|)', $template);
        $template = str_replace('/', '\/', $template);
        if (preg_match("/^{$template}$/", $path) === 1) {
            $ok = true;
            break 2;
        }
    }
}

if (!$ok) {
    $response->set_code(400);
    $response->set_reason("No handler found for {$path} {$accept} {$contenttype}");
} else {
    $body = file_get_contents('php://input');
    $response->set_request_data($body);
    if (in_array($response->get_request_method(), $resource->get_methods())) {
        $resource->execute($response);
    } else {
        $response->set_code(405);
    }
}

// -----------------------------------------------------------
//  NEW: Extract LTI ID & Log It
// -----------------------------------------------------------
global $DB;

$ltiid = 0;
$matches = [];
if (preg_match('/services\.php\/(\d+)\//', $_SERVER['REQUEST_URI'], $matches)) {
    $ltiid = intval($matches[1]);
} elseif (preg_match('/services\.php\/(\d+)($|\/)/', $_SERVER['REQUEST_URI'], $matches)) {
    $ltiid = intval($matches[1]);
}

if ($ltiid <= 0) {
    file_put_contents($logfile, date('Y-m-d H:i:s') . " | ERROR: LTI ID not found | URI: " . $_SERVER['REQUEST_URI'] . "\n", FILE_APPEND);
} else {
    file_put_contents($logfile, date('Y-m-d H:i:s') . " | Found LTI ID: {$ltiid}\n", FILE_APPEND);

    // Get module + course mapping
    $moduleid = $DB->get_field('modules', 'id', ['name' => 'lti']);
    if (empty($moduleid)) {
        file_put_contents($logfile, date('Y-m-d H:i:s') . " | ERROR: 'lti' module not found.\n", FILE_APPEND);
    } else {
        $cmid = $DB->get_field('course_modules', 'id', ['instance' => $ltiid, 'module' => $moduleid]);
        if (empty($cmid)) {
            file_put_contents($logfile, date('Y-m-d H:i:s') . " | ERROR: No course_modules entry for instance={$ltiid} module={$moduleid}\n", FILE_APPEND);
        } else {
            $courseid = $DB->get_field('course_modules', 'course', ['id' => $cmid]);
            if (empty($courseid)) {
                file_put_contents($logfile, date('Y-m-d H:i:s') . " | ERROR: Course not found for course_modules.id={$cmid}\n", FILE_APPEND);
            } else {
                file_put_contents($logfile, date('Y-m-d H:i:s') . " | Course={$courseid} | CMID={$cmid}\n", FILE_APPEND);
            }
        }
    }
}

// -----------------------------------------------------------
//  LTI SCORE HANDLER + GET RESPONSE FOR AWS
// -----------------------------------------------------------
$rawdata = file_get_contents('php://input');
$decoded = json_decode($rawdata, true);

// === Handle GET /lineitems ===
if ($_SERVER['REQUEST_METHOD'] === 'GET' && strpos($_SERVER['REQUEST_URI'], '/lineitems') !== false) {
    $lineitemurl = "https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

    $lineitems = [[
        "id" => $lineitemurl,
        "scoreMaximum" => 100,
        "label" => "AWS Academy Lab",
        "resourceId" => "aws-lab",
        "tag" => "AWS Course",
        "resourceLinkId" => "lti-resource",
        "gradesReleased" => true
    ]];

    header('Content-Type: application/vnd.ims.lis.v2.lineitemcontainer+json');
    echo json_encode($lineitems);
    exit;
}

// === Handle GET /lineitem ===
if ($_SERVER['REQUEST_METHOD'] === 'GET' && strpos($_SERVER['REQUEST_URI'], '/lineitem') !== false) {
    $lineitemurl = "https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

    $lineitem = [
        "id" => $lineitemurl,
        "scoreMaximum" => 100,
        "label" => "AWS Academy Lab",
        "resourceId" => "aws-lab",
        "tag" => "AWS Course",
        "resourceLinkId" => "lti-resource",
        "gradesReleased" => true
    ];

    header('Content-Type: application/vnd.ims.lis.v2.lineitem+json');
    echo json_encode($lineitem);
    exit;
}

// === Handle POST /scores ===
if (!empty($decoded) && isset($decoded['userId'])) {
    $userid = intval($decoded['userId']);
    $score = isset($decoded['scoreGiven']) ? floatval($decoded['scoreGiven']) : null;
    $scoremax = isset($decoded['scoreMaximum']) ? floatval($decoded['scoreMaximum']) : null;

    // ✅ Restrict activityProgress
    $activityprogress = !empty($decoded['activityProgress']) ? $decoded['activityProgress'] : 'Unknown';
    $allowedprogress = ['Started', 'InProgress', 'Completed'];
    if (!in_array($activityprogress, $allowedprogress)) {
        $activityprogress = 'Started';
    }

    $gradingprogress = !empty($decoded['gradingProgress']) ? $decoded['gradingProgress'] : 'NotReady';

    $existing = $DB->get_record('local_ltiprogress', ['userid' => $userid, 'ltiid' => $ltiid]);
    $record = new stdClass();
    $record->userid = $userid;
 
$record->courseid = $courseid;      
$record->cmid     = $cmid ?? 0;
$record->ltiid    = $ltiid;  

    $record->activityprogress = $activityprogress;
    $record->gradingprogress = $gradingprogress;
    $record->score = $score;
    $record->scoremax = $scoremax;
    $record->timemodified = time();

   // 🔥 Freeze logic: Started → InProgress → Completed, never downgrade

if ($existing) {

    // Normalize old + new values
    $old = strtolower($existing->activityprogress);
    $new = strtolower($activityprogress);

    // Define numeric levels
    $levels = [
        'started'    => 1,
        'inprogress' => 2,
        'completed'  => 3
    ];

    $oldlevel = $levels[$old] ?? 1;
    $newlevel = $levels[$new] ?? 1;

    // If already completed or new level is lower → DO NOT UPDATE
    if ($oldlevel >= 3 || $newlevel < $oldlevel) {

        file_put_contents($logfile,
            date('Y-m-d H:i:s') . " | SKIP UPDATE: OLD={$old} NEW={$new} USER={$userid}\n",
            FILE_APPEND
        );

    } else {
        // Allowed upgrade (Started→InProgress or InProgress→Completed)
        $record->id = $existing->id;
        $DB->update_record('local_ltiprogress', $record);

        file_put_contents($logfile,
            date('Y-m-d H:i:s') . " | UPDATED: {$old} → {$new} USER={$userid}\n",
            FILE_APPEND
        );
    }

} else {
    // First time insert always allowed
    $record->timecreated = time();
    $DB->insert_record('local_ltiprogress', $record);

    file_put_contents($logfile,
        date('Y-m-d H:i:s') . " | INSERT: NEW ROW {$activityprogress} USER={$userid}\n",
        FILE_APPEND
    );
}


    if (!is_null($score) && !is_null($scoremax)) {
        require_once($CFG->libdir . '/gradelib.php');
        $params = [
            'userid' => $userid,
            'rawgrade' => $score,
            'rawgrademax' => $scoremax
        ];
        grade_update('mod/lti', $courseid, 'mod', 'lti', $ltiid, 0, [$userid => $params]);
    }

    if ($activityprogress === 'Completed' || $gradingprogress === 'FullyGraded') {
        $completion = $DB->get_record('course_modules_completion', ['userid' => $userid, 'coursemoduleid' => $cmid]);
        if ($completion) {
            $completion->completionstate = 1;
            $completion->timemodified = time();
            $DB->update_record('course_modules_completion', $completion);
        } else {
            $completion = new stdClass();
            $completion->coursemoduleid = $cmid;
            $completion->userid = $userid;
            $completion->completionstate = 1;
            $completion->timemodified = time();
            $DB->insert_record('course_modules_completion', $completion);
        }
    }

    // Log successful DB save in lti-debug.log
    file_put_contents(
        $logfile,
        date('Y-m-d H:i:s') . " | USER: {$userid} | LTI: {$ltiid} | SCORE: {$score} | STATUS: {$activityprogress} / {$gradingprogress}\n",
        FILE_APPEND
    );
}

$response->send();
