<?php
require_once(__DIR__ . '/../../config.php');
require_login();

global $DB, $USER, $PAGE, $OUTPUT;

// Page setup
$PAGE->set_url(new moodle_url('/local/custompage/training2.php'));
$PAGE->set_context(context_system::instance());
$PAGE->set_pagelayout('standard');
$PAGE->set_title('Learning Path');
$PAGE->set_heading('Learning Path');

echo $OUTPUT->header();

/* Pagination */
$perpage = 10;
$page = optional_param('page', 0, PARAM_INT);
$start = $page * $perpage;
$today = time();

/* ---------------------------------------------------
 * 1️⃣ Get all PUBLISHED LPs where user is enrolled
 * --------------------------------------------------- */
$sql = "
    SELECT DISTINCT lp.*
      FROM {learning_paths} lp
      JOIN {learning_path_enrolments} e
        ON e.learningpathid = lp.id
     WHERE e.userid = :userid
       AND lp.status = 'Publish'
  ORDER BY lp.id DESC
";

$allpaths = $DB->get_records_sql($sql, ['userid' => $USER->id]);

/* ---------------------------------------------------
 * 2️⃣ FILTER LPs BASED ON COHORT + USER PROGRESS
 * --------------------------------------------------- */
$filtered = [];

foreach ($allpaths as $lp) {

    /* 🔹 Check if LP is STARTED by user
       Started = at least 1 course completed OR in-progress */
    $started = $DB->record_exists_sql("
        SELECT 1
          FROM {learning_path_courses} c
          JOIN {course_completions} cc
            ON cc.course = c.courseid
           AND cc.userid = ?
         WHERE c.sectionid IN (
                SELECT id
                  FROM {learning_path_sections}
                 WHERE learningpathid = ?
               )
           AND (
                cc.timecompleted IS NOT NULL
                OR (cc.timestarted IS NOT NULL AND cc.timecompleted IS NULL)
           )
    ", [$USER->id, $lp->id]);

    /* 🔹 Check cohort visibility */
    $enrols = $DB->get_records('learning_path_enrolments', [
        'learningpathid' => $lp->id
    ]);

    $hide = false;

    foreach ($enrols as $en) {

        if (!empty($en->cohortid)) {

            $cohort = $DB->get_record('cohort', ['id' => $en->cohortid]);

            if ($cohort) {

                $cohorthidden =
                    ($cohort->visible == 0) ||
                    (!empty($cohort->end_date) && $cohort->end_date < $today);

                // 🚨 HIDE ONLY IF cohort hidden AND user never started LP
                if ($cohorthidden && !$started) {
                    $hide = true;
                    break;
                }
            }
        }
    }

    if (!$hide) {
        $filtered[$lp->id] = $lp;
    }
}

/* ---------------------------------------------------
 * 3️⃣ Pagination after filtering
 * --------------------------------------------------- */
$total = count($filtered);
$paths = array_slice($filtered, $start, $perpage, true);
?>

<style>
:root {
    --primary:#FF4B00;
    --secondary:#F16521;
    --light-bg:#FFF5ED;
    --card-bg:#ffffff;
    --border:#f1e1d8;
    --text-dark:#2b1f1a;
}
.container { max-width:1100px !important; }
h3 { color:var(--primary); font-weight:700; margin-bottom:20px; }

.lp-item {
    background:var(--card-bg);
    padding:18px 20px;
    border:1px solid var(--border);
    border-radius:12px;
    margin-bottom:15px;
    box-shadow:0 2px 8px rgba(0,0,0,0.05);
}
.lp-name { font-size:18px; font-weight:700; color:var(--text-dark); }

.lp-progress {
    width:230px; height:8px; background:#ffe1d3;
    border-radius:6px; overflow:hidden; margin-top:8px;
}
.lp-progress-fill {
    height:8px; background:var(--primary);
}

.access-btn {
    background:var(--primary); color:#fff;
    padding:6px 16px; border-radius:6px;
    text-decoration:none; font-size:14px; font-weight:600;
}
.toggle-btn {
    background:var(--light-bg);
    border:1px solid var(--secondary);
    color:var(--primary);
    cursor:pointer; border-radius:6px;
    font-size:13px; padding:2px 8px;
}
.course-item { margin:3px 0 3px 30px; font-size:13px; }
</style>

<div class="container">
<h3>Learning Path</h3>

<?php if (empty($paths)): ?>
    <div style="padding:20px;">You are not enrolled in any learning path.</div>
<?php endif; ?>

<?php foreach ($paths as $lp):

    /* 🔹 Progress calculation */
    $completed = $DB->count_records_sql("
        SELECT COUNT(*)
          FROM {learning_path_courses} c
          JOIN {course_completions} cc
            ON cc.course = c.courseid
           AND cc.userid = ?
         WHERE c.sectionid IN (
                SELECT id
                  FROM {learning_path_sections}
                 WHERE learningpathid = ?
               )
           AND cc.timecompleted IS NOT NULL
    ", [$USER->id, $lp->id]);

    $totalcourses = $DB->count_records_sql("
        SELECT COUNT(*)
          FROM {learning_path_courses}
         WHERE sectionid IN (
                SELECT id
                  FROM {learning_path_sections}
                 WHERE learningpathid = ?
               )
    ", [$lp->id]);

    $progress = $totalcourses > 0 ? round(($completed / $totalcourses) * 100) : 0;
?>

<div class="lp-item">
    <div style="display:flex; justify-content:space-between; align-items:center;">
        <div>
            <div class="lp-name"><?= format_string($lp->name) ?></div>
            <div class="lp-progress">
                <div class="lp-progress-fill" style="width:<?= $progress ?>%"></div>
            </div>
        </div>

        <div style="display:flex; gap:10px;">
            <a class="access-btn"
               href="<?= new moodle_url('/local/learningpath/learner_path_view.php', ['id'=>$lp->id]) ?>">
               Access Now
            </a>
            <button class="toggle-btn"
                onclick="toggleSections(<?= $lp->id ?>)"
                id="btn-lp-<?= $lp->id ?>">▶</button>
        </div>
    </div>

    <div id="sections-<?= $lp->id ?>" style="display:none; margin-top:10px;">
        <?php
        $sections = $DB->get_records('learning_path_sections',
            ['learningpathid'=>$lp->id], 'id ASC');

        foreach ($sections as $sec):
        ?>
            <div style="margin-top:10px;">
                <button class="toggle-btn"
                    onclick="toggleCourses(<?= $sec->id ?>)"
                    id="btn-sec-<?= $sec->id ?>">▶</button>
                <strong><?= format_string($sec->sectionname) ?></strong>
            </div>

            <div id="courses-<?= $sec->id ?>" style="display:none;">
                <?php
                $courses = $DB->get_records('learning_path_courses',
                    ['sectionid'=>$sec->id]);
                foreach ($courses as $c):
                    if ($course = $DB->get_record('course', ['id'=>$c->courseid])):
                ?>
                    <div class="course-item">• <?= format_string($course->fullname) ?></div>
                <?php endif; endforeach; ?>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<?php endforeach; ?>

<?= $OUTPUT->paging_bar($total, $page, $perpage,
    new moodle_url('/local/custompage/training2.php')) ?>

</div>

<script>
function toggleSections(id){
    let s=document.getElementById("sections-"+id);
    let b=document.getElementById("btn-lp-"+id);
    if(s.style.display==="none"){ s.style.display="block"; b.innerHTML="▼"; }
    else{ s.style.display="none"; b.innerHTML="▶"; }
}
function toggleCourses(id){
    let s=document.getElementById("courses-"+id);
    let b=document.getElementById("btn-sec-"+id);
    if(s.style.display==="none"){ s.style.display="block"; b.innerHTML="▼"; }
    else{ s.style.display="none"; b.innerHTML="▶"; }
}
</script>

<?php
echo $OUTPUT->footer();
