<?php
require_once(__DIR__ . '/../config.php');
require_login();

global $DB, $CFG, $OUTPUT, $USER;

$PAGE->set_context(context_system::instance());
$PAGE->set_url(new moodle_url('/course/scompany.php'));
$PAGE->set_pagelayout('standard');
$PAGE->set_title('Add / Edit Company');
$PAGE->set_heading('Add / Edit Company');

// ---------------------------------------------
// GET EDIT ID
// ---------------------------------------------
$editid = optional_param('id', 0, PARAM_INT);

// ---------------------------------------------
// ROLE CHECK: IS LOGGED USER A MANAGER?
// ---------------------------------------------
$manager_role = $DB->get_record('role', ['shortname' => 'manager']);
$ismanager = false;

if ($manager_role) {
    $ismanager = user_has_role_assignment($USER->id, $manager_role->id);
}

// ---------------------------------------------
// FETCH MANAGERS FOR DROPDOWN (ONLY FOR ADMINS)
// ---------------------------------------------
$manageroptions = [0 => "Select Manager"];

if (!$ismanager && $manager_role) {
    $managers = $DB->get_records_sql("
        SELECT DISTINCT u.id, u.firstname, u.lastname
        FROM {role_assignments} ra
        JOIN {user} u ON u.id = ra.userid
        WHERE ra.roleid = :rid
        ORDER BY u.firstname ASC
    ", ['rid' => $manager_role->id]);

    foreach ($managers as $m) {
        $manageroptions[$m->id] = "{$m->firstname} {$m->lastname}";
    }
}

// ---------------------------------------------
// LOAD DATA IF EDIT
// ---------------------------------------------
$company = null;
if ($editid > 0) {
    $company = $DB->get_record('company', ['id' => $editid], '*', MUST_EXIST);
}

// ---------------------------------------------
// PROCESS FORM SUBMIT
// ---------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && confirm_sesskey()) {

    $data = new stdClass();
    $data->com_name = required_param('com_name', PARAM_TEXT);

    // -------- MANAGER ASSIGNMENT LOGIC --------
    if ($ismanager) {
        // Manager logged in → force manager ID
        $data->class_manager = $USER->id;
    } else {
        // Admin logged in → admin chooses manager
        $data->class_manager = required_param('class_manager', PARAM_INT);
    }
    // ------------------------------------------

    $data->email   = optional_param('email', '', PARAM_EMAIL);
    $data->phone   = optional_param('phone', '', PARAM_RAW);
    $data->website = optional_param('website', '', PARAM_RAW);
    $data->address = optional_param('address', '', PARAM_TEXT);
    $data->zip     = optional_param('zip', '', PARAM_TEXT);

    if ($editid > 0) {
        $data->id = $editid;
        $DB->update_record('company', $data);
        redirect($PAGE->url, "Company updated successfully", 2);
    } else {
        $data->created_at = time();
        $DB->insert_record('company', $data);
        redirect($PAGE->url, "Company added successfully", 2);
    }
}

echo $OUTPUT->header();
?>

<!-- ------------------ CSS ------------------ -->
<style>
    #page-header { margin-top: 0px !important; }
    #region-main { padding: 0px 0px !important; }

    .company-form-container {
        width: 100%;
        max-width: 1000px;
        margin: 20px auto;
        background: #fff;
        border-radius: 10px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.1);
        font-family: "Segoe UI", sans-serif;
        padding: 10px;
    }

    .company-form-header {
        background-color: #f7f7f7;
        padding: 20px 30px;
        font-size: 22px;
        font-weight: bold;
        color: #2c3e50;
        border-bottom: 1px solid #ddd;
    }

    .company-form-body {
        padding: 30px;
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 25px;
    }

    .form-group label { font-weight: 600; margin-bottom: 6px; }
    .form-group input, .form-group select, .form-group textarea {
        padding: 12px; border: 1px solid #ccc;
        border-radius: 6px; font-size: 15px; width: 100%;
    }
    textarea { resize: vertical; height: 100px; }

    .company-form-footer {
        grid-column: 1 / -1;
        text-align: right;
    }
</style>

<div class="company-form-container">
    <div class="company-form-header">
        <?php echo ($editid > 0 ? "Edit Company" : "Add Company"); ?>
    </div>

    <form method="post">
        <input type="hidden" name="sesskey" value="<?php echo sesskey(); ?>">

        <div class="company-form-body">

            <!-- Company Name -->
            <div class="form-group">
                <label>Company Name</label>
                <input type="text" name="com_name" required
                    value="<?php echo ($company->com_name ?? ''); ?>">
            </div>

            <!-- Manager Field -->
            <?php if ($ismanager) { ?>

                <!-- Manager sees their name (read only) -->
                <div class="form-group">
                    <label>Manager</label>
                    <input type="text" value="<?php echo $USER->firstname . ' ' . $USER->lastname; ?>" disabled>
                    <input type="hidden" name="class_manager" value="<?php echo $USER->id; ?>">
                </div>

            <?php } else { ?>

                <!-- Admin sees dropdown -->
                <div class="form-group">
                    <label>Manager</label>
                    <select name="class_manager">
                        <?php foreach ($manageroptions as $id => $name) { ?>
                            <option value="<?php echo $id; ?>"
                                <?php if (!empty($company->class_manager) && $company->class_manager == $id) echo 'selected'; ?>>
                                <?php echo $name; ?>
                            </option>
                        <?php } ?>
                    </select>
                </div>

            <?php } ?>

            <!-- Email -->
            <div class="form-group">
                <label>Email</label>
                <input type="email" name="email"
                    value="<?php echo ($company->email ?? ''); ?>">
            </div>

            <!-- Phone -->
            <div class="form-group">
                <label>Phone</label>
                <input type="text" name="phone"
                    value="<?php echo ($company->phone ?? ''); ?>">
            </div>

            <!-- Website -->
            <div class="form-group">
                <label>Website</label>
                <input type="text" name="website"
                    value="<?php echo ($company->website ?? ''); ?>">
            </div>

            <!-- Zip -->
            <div class="form-group">
                <label>Zip Code</label>
                <input type="text" name="zip"
                    value="<?php echo ($company->zip ?? ''); ?>">
            </div>

            <!-- Address -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label>Address</label>
                <textarea name="address"><?php echo ($company->address ?? ''); ?></textarea>
            </div>

        </div>

        <div class="company-form-footer">
            <button type="submit" class="btn btn-primary">
                <?php echo ($editid > 0 ? "Update Company" : "Add Company"); ?>
            </button>
        </div>

    </form>
</div>

<?php echo $OUTPUT->footer(); ?>
