<?php
// Place this file inside: moodle/Trainocate/aws_cleanup_newest.php
// Run: php aws_cleanup_newest.php        → dry run
//      php aws_cleanup_newest.php --apply  → deletes old LTIs

define('CLI_SCRIPT', true);

// Load Moodle config
require(__DIR__ . '/config.php');
require_once($CFG->dirroot . '/course/lib.php');
require_once($CFG->dirroot . '/course/modlib.php');

global $DB;

$apply = in_array('--apply', $argv) || in_array('-a', $argv);

echo "\n=============================\n";
echo " AWS LTI CLEANUP (KEEP NEWEST)\n";
echo "=============================\n";
echo $apply ? "MODE: APPLY (deletions will occur)\n" : "MODE: DRY RUN (no deletions)\n\n";

// 1) Find all courses that have more than one LTI entry
$sql = "SELECT course, COUNT(*) AS total
        FROM {lti}
        GROUP BY course
        HAVING COUNT(*) > 1";

$courses = $DB->get_records_sql($sql);

if (!$courses) {
    echo "No courses with duplicate LTI found.\n";
    exit;
}

foreach ($courses as $c) {

    $courseid = $c->course;
    $course = $DB->get_record('course', ['id' => $courseid], '*', MUST_EXIST);

    echo "\n-----------------------------------------\n";
    echo " Course: {$course->id} - {$course->shortname}\n";
    echo "-----------------------------------------\n";

    // 2) Get all LTI modules (newest first)
    $sql2 = "SELECT l.*, cm.id AS cmid
             FROM {lti} l
             JOIN {course_modules} cm ON cm.instance = l.id
             JOIN {modules} m ON m.id = cm.module
             WHERE m.name = 'lti' AND l.course = ?
             ORDER BY l.timemodified DESC, l.timecreated DESC";

    $ltis = $DB->get_records_sql($sql2, [$courseid]);

    if (count($ltis) <= 1) {
        echo " Only one LTI → skip.\n";
        continue;
    }

    // Convert to array
    $ltis_list = array_values($ltis);

    // Newest LTI (keep this one)
    $keep = $ltis_list[0];
    echo " ✔ Keeping newest LTI: id={$keep->id}, cmid={$keep->cmid}\n";

    // Older LTIs
    $to_delete = array_slice($ltis_list, 1);

    if (!$apply) {
        foreach ($to_delete as $old) {
            echo "   → WOULD DELETE old LTI: id={$old->id}, cmid={$old->cmid}\n";
        }
        continue; // skip deletion
    }

    // APPLY MODE — delete old LTIs
    $tx = $DB->start_delegated_transaction();

    try {
        foreach ($to_delete as $old) {
            echo "   → DELETING old LTI: id={$old->id}, cmid={$old->cmid}\n";
            course_delete_module($old->cmid); // Safely removes CM + instance
        }

        // 3) Fix duplicate section sequences
        $sections = $DB->get_records('course_sections', ['course' => $courseid]);

        foreach ($sections as $sec) {

            if (empty($sec->sequence)) continue;

            $seq = explode(',', $sec->sequence);

            // Remove duplicates
            $uniq = array_unique($seq);
            $newSeq = implode(',', $uniq);

            if ($newSeq !== $sec->sequence) {
                $sec->sequence = $newSeq;
                $DB->update_record('course_sections', $sec);
                echo "   ✔ Cleaned duplicate section sequence (section {$sec->section})\n";
            }
        }

        // 4) Rebuild cache
        rebuild_course_cache($courseid, true);
        echo " ✔ Cache rebuilt.\n";

        $tx->allow_commit();

    } catch (Exception $e) {
        $tx->rollback($e);
        echo " ❌ ERROR: " . $e->getMessage() . "\n";
        continue;
    }
}

echo "\n===== DONE =====\n";
echo $apply ? "Cleanup applied successfully.\n" : "Dry-run completed. Use --apply to execute.\n";
