<?php
require_once(__DIR__ . '/../../config.php');
require_login();
global $DB, $PAGE, $OUTPUT;

$context = context_system::instance();
require_capability('moodle/site:config', $context);

$PAGE->set_url(new moodle_url('/local/bulk_enrolmanager/index.php'));
$PAGE->set_context($context);
$PAGE->set_title('Bulk Enrolment Manager');
$PAGE->set_heading('Bulk Enrolment Manager');

echo $OUTPUT->header();
// ✅ Load Select2 (for searchable dropdowns)
echo '
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
';

/* --------------------------
   ✅ STYLING (FULL WIDTH + CLEAN UI)
--------------------------- */
echo '<style>
.bulk-enrol-container {
    width: 100%;
    max-width: 98%;
    margin: 20px auto;
    background: #fff;
    padding: 30px;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.bulk-enrol-container h2 {
    font-size: 24px;
    margin-bottom: 25px;
    border-bottom: 2px solid #0073aa;
    padding-bottom: 10px;
    color: #0073aa;
}

.bulk-enrol-filter {
    display: flex;
    flex-wrap: wrap;
    gap: 20px;
    margin-bottom: 25px;
    align-items: flex-end;
    justify-content: flex-start;
    width: 100%;
}

.bulk-enrol-filter > div {
    flex: 1;
    min-width: 300px;
}

.bulk-enrol-filter label {
    font-weight: 600;
    margin-bottom: 5px;
    display: block;
    color: #333;
}

.bulk-enrol-filter select {
    width: 100%;
    padding-left: 10px;
    border-radius: 6px;
    border: 1px solid #ccc;
    font-size: 15px;
}

.bulk-enrol-filter input[type="submit"] {
    background: #0073aa;
    color: #fff;
    border: none;
    padding: 10px 25px;
    border-radius: 6px;
    cursor: pointer;
}
.bulk-enrol-filter input[type="submit"]:hover {
    background: #005f8d;
}

.bulk-enrol-form label {
    display: block;
    margin-top: 15px;
    font-weight: 600;
    color: #333;
}

.bulk-enrol-form select {
    display: block;
    width: 100% !important;
    max-width: 100% !important;
    padding: 12px;
    border-radius: 6px;
    border: 1px solid #ccc;
    font-size: 15px;
    box-sizing: border-box;
}

.bulk-enrol-form select[multiple] {
    height: 250px;
}

.bulk-enrol-form input[type="submit"] {
    margin-top: 25px;
    background: #0073aa;
    color: #fff;
    border: none;
    padding: 12px 25px;
    border-radius: 6px;
    font-size: 16px;
    cursor: pointer;
    display: block;
    width: 100%;
}

.bulk-enrol-form input[type="submit"]:hover {
    background: #005f8d;
}
</style>';

/* --------------------------
   ✅ PAGE CONTENT START
--------------------------- */
echo '<div class="bulk-enrol-container">';
echo '<h2>Bulk Enrolment Manager</h2>';

/* --------------------------
   ✅ SUBMIT HANDLER
--------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && confirm_sesskey()) {

    // ✅ Get parameters safely first
    $courses = optional_param_array('courses', [], PARAM_INT);
    $users   = optional_param_array('users', [], PARAM_INT);
    $batches = optional_param_array('batches', [], PARAM_INT);

    $courses = array_filter($courses);
    $users   = array_filter($users);
    $batches = array_filter($batches);

    if (!empty($courses) && (!empty($users) || !empty($batches))) {
        require_once($CFG->dirroot . '/enrol/manual/lib.php');
        $enrol = enrol_get_plugin('manual');

        // Get the Student role dynamically (instead of hard-coded 5)
        $studentroleid = $DB->get_field('role', 'id', ['shortname' => 'student']);

        foreach ($courses as $courseid) {
            // ✅ Get manual enrolment instance for the course
            $manualinstance = null;
            if ($instances = enrol_get_instances($courseid, false)) {
                foreach ($instances as $instance) {
                    if ($instance->enrol === 'manual') {
                        $manualinstance = $instance;
                        break;
                    }
                }
            }

            // ⚠️ If no manual instance, skip this course
            if (empty($manualinstance)) {
                continue;
            }

            // ✅ Enrol selected users
            foreach ($users as $userid) {
                if (empty($userid)) {
                    continue; // Skip invalid
                }

                $existing = $DB->record_exists_sql("
                    SELECT 1
                    FROM {user_enrolments} ue
                    JOIN {enrol} e ON ue.enrolid = e.id
                    WHERE e.id = :enrolid AND ue.userid = :userid
                ", ['enrolid' => $manualinstance->id, 'userid' => $userid]);

                if (!$existing) {
                    $enrol->enrol_user($manualinstance, $userid, $studentroleid);
                }
            }

            // ✅ Enrol users from batches (cohorts)
            foreach ($batches as $cohortid) {
                if (empty($cohortid)) {
                    continue;
                }

                $cohortmembers = $DB->get_records('cohort_members', ['cohortid' => $cohortid], '', 'userid');
                foreach ($cohortmembers as $member) {
                    $userid = $member->userid;

                    if (empty($userid)) {
                        continue;
                    }

                    $existing = $DB->record_exists_sql("
                        SELECT 1
                        FROM {user_enrolments} ue
                        JOIN {enrol} e ON ue.enrolid = e.id
                        WHERE e.id = :enrolid AND ue.userid = :userid
                    ", ['enrolid' => $manualinstance->id, 'userid' => $userid]);

                    if (!$existing) {
                        $enrol->enrol_user($manualinstance, $userid, $studentroleid);
                    }
                }
            }
        }

        // ✅ Redirect with success message
        redirect(
            new moodle_url('/local/bulk_enrolmanager/index.php'),
            '✅ Enrolment completed successfully!',
            null,
            \core\output\notification::NOTIFY_SUCCESS
        );
    }
}


/* --------------------------
   ✅ LOAD DOMAIN OPTIONS
--------------------------- */
$domainfield = $DB->get_record('customfield_field', ['shortname' => 'domain']);
$domainoptions = [];
if ($domainfield) {
    $domainoptions = $DB->get_records_sql_menu("
        SELECT DISTINCT value AS id, value
        FROM {customfield_data}
        WHERE fieldid = :fid AND value IS NOT NULL AND value <> ''
        ORDER BY value ASC
    ", ['fid' => $domainfield->id]);
}

/* --------------------------
   ✅ LANGUAGE MAPPING (AWS codes)
--------------------------- */
$lang_names = [
    'ar-SA' => 'Arabic (Saudi Arabia)',
    'de-DE' => 'German (Germany)',
    'en-US' => 'English (United States)',
    'es-419' => 'Spanish (Latin America)',
    'es-ES' => 'Spanish (Spain)',
    'fr-FR' => 'French (France)',
    'id-ID' => 'Indonesian',
    'it-IT' => 'Italian',
    'ja-JP' => 'Japanese',
    'ko-KR' => 'Korean',
    'pt-BR' => 'Portuguese (Brazil)',
    'zh-CN' => 'Chinese (Simplified)',
    'zh-TW' => 'Chinese (Traditional)',
    'th-TH' => 'Thai',
    'vi-VN' => 'Vietnamese',
    'he-IL' => 'Hebrew',
    'el-GR' => 'Greek',
    'pl-PL' => 'Polish',
    'tr-TR' => 'Turkish',
    'uk-UA' => 'Ukrainian'
];

/* --------------------------
   ✅ LOAD LANGUAGE OPTIONS (from course table)
--------------------------- */
$rawlangs = $DB->get_records_sql_menu("
    SELECT DISTINCT lang AS id, lang
    FROM {course}
    WHERE lang IS NOT NULL AND lang <> ''
    ORDER BY lang ASC
");

$languageoptions = [];
foreach ($rawlangs as $code => $val) {
    $languageoptions[$code] = $lang_names[$code] ?? strtoupper($code);
}

/* --------------------------
   ✅ READ FILTER VALUES
--------------------------- */
$selecteddomain = optional_param('domainfilter', '', PARAM_TEXT);
$selectedlanguage = optional_param('languagefilter', '', PARAM_TEXT);
/* --------------------------
   ✅ FILTER FORM (with All option inside dropdown)
--------------------------- */
echo '<form method="get" class="bulk-enrol-filter">';

// Add "All" at the beginning of each array
$domainoptions = ['all' => 'All Domains'] + $domainoptions;
$languageoptions = ['all' => 'All Languages'] + $languageoptions;

echo '<div><label>Filter by Domain:</label>';
echo html_writer::select($domainoptions, 'domainfilter', $selecteddomain ?: 'all', null, ['class' => 'searchable-select']);
echo '</div>';

echo '<div><label>Filter by Language:</label>';
echo html_writer::select($languageoptions, 'languagefilter', $selectedlanguage ?: 'all', null, ['class' => 'searchable-select']);
echo '</div>';

echo '<div style="flex:0 0 auto;"><input type="submit" value="Apply Filter" /></div>';
echo '</form>';


/* --------------------------
   ✅ FILTER COURSES BY DOMAIN & LANGUAGE
--------------------------- */
$coursefilter = [];
$params = [];

if ($selecteddomain !== '' && $selecteddomain !== 'all') {

    $coursefilter[] = "c.id IN (
        SELECT instanceid FROM {customfield_data}
        WHERE fieldid = :domainfid AND value = :domainval
    )";
    $params['domainfid'] = $domainfield->id;
    $params['domainval'] = $selecteddomain;
}

if ($selectedlanguage !== '' && $selectedlanguage !== 'all') {

    $coursefilter[] = "c.lang = :langval";
    $params['langval'] = $selectedlanguage;
}

$where = !empty($coursefilter) ? 'WHERE ' . implode(' AND ', $coursefilter) : '';

$courses = $DB->get_records_sql_menu("
    SELECT c.id, c.fullname
    FROM {course} c
    $where
    ORDER BY c.fullname ASC
", $params);

/* --------------------------
   ✅ USERS & COHORTS
--------------------------- */
$users = $DB->get_records_sql_menu("
    SELECT id, CONCAT(firstname, ' ', lastname, ' (', email, ')') AS fullname
    FROM {user}
    WHERE deleted = 0 AND suspended = 0 AND username NOT IN ('guest', 'admin')
    ORDER BY firstname ASC
");

$batches = $DB->get_records_menu('cohort', null, 'name ASC', 'id, name');

/* --------------------------
   ✅ MAIN FORM
--------------------------- */
echo '<form method="post" class="bulk-enrol-form">';
echo '<input type="hidden" name="sesskey" value="'.sesskey().'" />';

echo '<label>Select Courses:</label>';
echo html_writer::select($courses, 'courses[]', '', ['' => 'Select courses...'], [
    'multiple' => 'multiple', 'size' => 8, 'class' => 'searchable-select'
]);
echo '<label>Select Users:</label>';
echo html_writer::select($users, 'users[]', '', ['' => 'Select users...'], [
    'multiple' => 'multiple', 'size' => 10, 'class' => 'searchable-select'
]);

echo '<label>Select Batches:</label>';
echo html_writer::select($batches, 'batches[]', '', ['' => 'Select batches...'], [
    'multiple' => 'multiple', 'size' => 6, 'class' => 'searchable-select'
]);

echo '<input type="submit" value="Enroll Selected" />';
echo '</form>';

echo '</div>';

/* --------------------------
   ✅ AUTO-SUBMIT FILTERS
--------------------------- */
echo '<script>
document.querySelectorAll(".bulk-enrol-filter select").forEach(sel => {
    sel.addEventListener("change", () => sel.form.submit());
});
</script>';


/* --------------------------
   ✅ VALIDATION SCRIPT (FIXED & WORKING)
--------------------------- */
echo '<script>
document.addEventListener("DOMContentLoaded", function() {
    const form = document.querySelector(".bulk-enrol-form");

    form.addEventListener("submit", function(e) {
        // Get selects
        const courseSelect = document.querySelector("select[name=\'courses[]\']");
        const userSelect = document.querySelector("select[name=\'users[]\']");
        const batchSelect = document.querySelector("select[name=\'batches[]\']");

        // Get selected options
        const selectedCourses = Array.from(courseSelect.selectedOptions).map(o => o.value).filter(v => v !== "");
        const selectedUsers = Array.from(userSelect.selectedOptions).map(o => o.value).filter(v => v !== "");
        const selectedBatches = Array.from(batchSelect.selectedOptions).map(o => o.value).filter(v => v !== "");

        // Remove any previous messages
        document.querySelectorAll(".form-error").forEach(el => el.remove());

        let valid = true;

        // Course validation
        if (selectedCourses.length === 0) {
            e.preventDefault();
            valid = false;
            showError(courseSelect, "⚠️ Please select at least one course before enrolling.");
        }

        // User or batch validation
        if (selectedUsers.length === 0 && selectedBatches.length === 0) {
            e.preventDefault();
            valid = false;
            showError(batchSelect, "⚠️ Please select at least one user or one batch before enrolling.");
        }

        // Scroll to first error
        if (!valid) {
            const firstError = document.querySelector(".form-error");
            if (firstError) {
                firstError.scrollIntoView({ behavior: "smooth", block: "center" });
            }
            return false;
        }

        return true;
    });

    function showError(element, message) {
        const div = document.createElement("div");
        div.className = "form-error";
        div.style.color = "red";
        div.style.marginTop = "8px";
        div.style.fontWeight = "600";
        div.textContent = message;

        // Insert after select box
        element.insertAdjacentElement("afterend", div);
    }
});
</script>';


// ✅ Initialize Select2 (dynamic placeholders, prevent placeholder chip)
echo '
<script>
document.addEventListener("DOMContentLoaded", function() {
    const placeholders = {
        "courses[]": "Select courses...",
        "users[]": "Select users...",
        "batches[]": "Select batches..."
    };

    $(".searchable-select").each(function() {
        const name = $(this).attr("name");
        const placeholder = placeholders[name] || "Select option";
        const $select = $(this);

        $select.select2({
            placeholder: placeholder,
            allowClear: true,
            width: "100%",
            closeOnSelect: false
        });

        // 🔹 Remove placeholder ("") if it somehow gets selected
        $select.on("select2:select select2:unselect change", function() {
            const values = $select.val() || [];
            const clean = values.filter(v => v !== ""); // remove empty option
            if (clean.length !== values.length) {
                $select.val(clean).trigger("change.select2");
            }
        });
    });
});
</script>
';




echo $OUTPUT->footer();
?>
