<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Allows you to edit a users profile (controller)
 *
 * @package core_user
 */

require_once('../config.php');
require_once($CFG->libdir.'/gdlib.php');
require_once($CFG->libdir.'/adminlib.php');
require_once($CFG->dirroot.'/user/editadvanced_form.php');
require_once($CFG->dirroot.'/user/editlib.php');
require_once($CFG->dirroot.'/user/profile/lib.php');
require_once($CFG->dirroot.'/user/lib.php');
require_once($CFG->dirroot.'/webservice/lib.php');

$id     = optional_param('id', $USER->id, PARAM_INT);    // User id; -1 if creating new user.
$course = optional_param('course', SITEID, PARAM_INT);   // Course id (defaults to Site).
$returnto = optional_param('returnto', null, PARAM_ALPHA);  // Code determining where to return to after save.

$PAGE->set_url('/user/editadvanced.php', array('course' => $course, 'id' => $id));
$course = $DB->get_record('course', array('id' => $course), '*', MUST_EXIST);

if (!empty($USER->newadminuser)) {
    ignore_user_abort(true);
    $PAGE->set_course($SITE);
    $PAGE->set_pagelayout('maintenance');
} else {
    if ($course->id == SITEID) {
        require_login();
        $PAGE->set_context(context_system::instance());
    } else {
        require_login($course);
    }
    $PAGE->set_pagelayout('admin');
    $PAGE->add_body_class('limitedwidth');
}

if ($course->id == SITEID) {
    $coursecontext = context_system::instance();
} else {
    $coursecontext = context_course::instance($course->id);
}
$systemcontext = context_system::instance();

if ($id == -1) {
    // Creating new user.
    $user = new stdClass();
    $user->id = -1;
    $user->auth = 'manual';
    $user->confirmed = 1;
    $user->deleted = 0;
    $user->timezone = '99';
    require_capability('moodle/user:create', $systemcontext);
    admin_externalpage_setup('addnewuser', '', array('id' => -1));
    $PAGE->set_primary_active_tab('siteadminnode');
    $PAGE->navbar->add(get_string('addnewuser', 'moodle'), $PAGE->url);
} else {
    // Editing existing user.
    require_capability('moodle/user:update', $systemcontext);
    $user = $DB->get_record('user', array('id' => $id), '*', MUST_EXIST);

    // Load emp_id and custom company_id into the $user object for the form.
    $user->emp_id = $DB->get_field('user', 'emp_id', array('id' => $user->id));
    // If company_id exists in DB, expose it as both names to be safe.
    $user->company_id = $DB->get_field('user', 'company_id', array('id' => $user->id));
    $user->companyid = $user->company_id;

    $PAGE->set_context(context_user::instance($user->id));
    $PAGE->navbar->includesettingsbase = true;
    if ($user->id != $USER->id) {
        $PAGE->navigation->extend_for_user($user);
    } else {
        if ($node = $PAGE->navigation->find('myprofile', navigation_node::TYPE_ROOTNODE)) {
            $node->force_open();
        }
    }
}

if ($user->id != -1 and is_mnet_remote_user($user)) {
    redirect($CFG->wwwroot . "/user/view.php?id=$id&course={$course->id}");
}

if ($user->id != $USER->id and is_siteadmin($user) and !is_siteadmin($USER)) {
    print_error('useradmineditadmin');
}

if (isguestuser($user->id)) {
    print_error('guestnoeditprofileother');
}

if ($user->deleted) {
    echo $OUTPUT->header();
    echo $OUTPUT->heading(get_string('userdeleted'));
    echo $OUTPUT->footer();
    die;
}

// Load preferences and profile data.
useredit_load_preferences($user);
profile_load_data($user);

// Interests
$user->interests = core_tag_tag::get_item_tags_array('core', 'user', $id);

// If editing an existing user, prepare editor area for description.
if ($user->id !== -1) {
    $usercontext = context_user::instance($user->id);
    $editoroptions = array(
        'maxfiles'   => EDITOR_UNLIMITED_FILES,
        'maxbytes'   => $CFG->maxbytes,
        'trusttext'  => false,
        'forcehttps' => false,
        'context'    => $usercontext
    );

    $user = file_prepare_standard_editor($user, 'description', $editoroptions, $usercontext, 'user', 'profile', 0);
} else {
    $usercontext = null;
    $editoroptions = array(
        'maxfiles' => 0,
        'maxbytes' => 0,
        'trusttext' => false,
        'forcehttps' => false,
        'context' => $coursecontext
    );
}

// Prepare filemanager draft area for picture
$draftitemid = 0;
$filemanagercontext = $editoroptions['context'];
$filemanageroptions = array('maxbytes'       => $CFG->maxbytes,
                             'subdirs'        => 0,
                             'maxfiles'       => 1,
                             'accepted_types' => 'optimised_image');
file_prepare_draft_area($draftitemid, $filemanagercontext->id, 'user', 'newicon', 0, $filemanageroptions);
$user->imagefile = $draftitemid;

// Create form.
$userform = new user_editadvanced_form(new moodle_url($PAGE->url, array('returnto' => $returnto)), array(
    'editoroptions' => $editoroptions,
    'filemanageroptions' => $filemanageroptions,
    'user' => $user));

// Decide return URL
if ($returnto === 'profile') {
    if ($course->id != SITEID) {
        $returnurl = new moodle_url('/user/view.php', array('id' => $user->id, 'course' => $course->id));
    } else {
        $returnurl = new moodle_url('/user/profile.php', array('id' => $user->id));
    }
} else if ($user->id === -1) {
    $returnurl = new moodle_url("/admin/user.php");
} else {
    $returnurl = new moodle_url("/admin/user.php");
}

// Handle cancel
if ($userform->is_cancelled()) {
    redirect($returnurl);
} else if ($usernew = $userform->get_data()) {
    $usercreated = false;

    if (empty($usernew->auth)) {
        $authplugin = get_auth_plugin($user->auth);
        unset($usernew->auth);
    } else {
        $authplugin = get_auth_plugin($usernew->auth);
    }

    // Normalize
    $usernew->timemodified = time();
    $createpassword = false;

    // Auto-generate Employee ID if not provided
    if (empty($usernew->emp_id)) {
        do {
            $lastid = $DB->get_field_sql("
                SELECT MAX(CAST(SUBSTRING(emp_id, 8) AS UNSIGNED))
                FROM {user}
                WHERE emp_id LIKE 'EMP_ID_%'
            ");
            $nextid = ($lastid) ? $lastid + 1 : 1;
            $newempid = 'EMP_ID_' . $nextid;
            $exists = $DB->record_exists('user', ['emp_id' => $newempid]);
        } while ($exists);

        $usernew->emp_id = $newempid;
    }

    // Auto-generate firstname/lastname if missing
    if (empty($usernew->firstname)) {
        $base = 'First';
        $counter = 1;
        $newname = $base.$counter;
        while ($DB->record_exists('user', ['firstname' => $newname])) {
            $counter++;
            $newname = $base.$counter;
        }
        $usernew->firstname = $newname;
    }
    if (empty($usernew->lastname)) {
        $base = 'Last';
        $counter = 1;
        $newname = $base.$counter;
        while ($DB->record_exists('user', ['lastname' => $newname])) {
            $counter++;
            $newname = $base.$counter;
        }
        $usernew->lastname = $newname;
    }

    // If creating a new user
    if ($usernew->id == -1) {
        unset($usernew->id);
        $createpassword = !empty($usernew->createpassword);
        unset($usernew->createpassword);
        $usernew = file_postupdate_standard_editor($usernew, 'description', $editoroptions, null, 'user', 'profile', null);
        $usernew->mnethostid = $CFG->mnet_localhost_id;
        $usernew->confirmed  = 1;
        $usernew->timecreated = time();

        if ($authplugin->is_internal()) {
            if ($createpassword or empty($usernew->newpassword)) {
                $usernew->password = '';
            } else {
                $usernew->password = hash_internal_user_password($usernew->newpassword);
            }
        } else {
            $usernew->password = AUTH_PASSWORD_NOT_CACHED;
        }

        // IMPORTANT: map companyid (from form) to company_id (DB column) before insert
        if (isset($usernew->companyid)) {
            $usernew->company_id = (int)$usernew->companyid;
        }

        $usernew->id = user_create_user($usernew, false, false);
        $DB->set_field('user', 'createdby', $USER->id, ['id' => $usernew->id]);
        $DB->set_field('user', 'emp_id', $usernew->emp_id, array('id' => $usernew->id));
        // In case user_create_user did not copy custom company_id, ensure it's set.
        if (isset($usernew->company_id)) {
            $DB->set_field('user', 'company_id', $usernew->company_id, ['id' => $usernew->id]);
        }

        if (!$authplugin->is_internal() and $authplugin->can_change_password() and !empty($usernew->newpassword)) {
            if (!$authplugin->user_update_password($usernew, $usernew->newpassword)) {
                debugging(get_string('cannotupdatepasswordonextauth', 'error', $usernew->auth), DEBUG_NONE);
            }
        }
        $usercreated = true;
    } else {
        // Updating existing user
        $usernew = file_postupdate_standard_editor($usernew, 'description', $editoroptions, $usercontext, 'user', 'profile', 0);
        if (!$authplugin->user_update($user, $usernew)) {
            print_error('cannotupdateuseronexauth', '', '', $user->auth);
        }

        user_update_user($usernew, false, false);

        // Ensure emp_id and company_id get saved
        $DB->set_field('user', 'emp_id', $usernew->emp_id, array('id' => $usernew->id));
        if (isset($usernew->companyid)) {
            $DB->set_field('user', 'company_id', (int)$usernew->companyid, array('id' => $usernew->id));
        }

        if (!empty($usernew->newpassword)) {
            if ($authplugin->can_change_password()) {
                if (!$authplugin->user_update_password($usernew, $usernew->newpassword)) {
                    print_error('cannotupdatepasswordonextauth', '', '', $usernew->auth);
                }
                unset_user_preference('create_password', $usernew);
                if (!empty($CFG->passwordchangelogout)) {
                    \core\session\manager::kill_user_sessions($usernew->id, session_id());
                }
                if (!empty($usernew->signoutofotherservices)) {
                    webservice::delete_user_ws_tokens($usernew->id);
                }
            }
        }

        if (isset($usernew->suspended) and $usernew->suspended and !$user->suspended) {
            \core\session\manager::kill_user_sessions($user->id);
        }
    }

    // Update preferences, tags, picture, etc.
    $usercontext = context_user::instance($usernew->id);
    useredit_update_user_preference($usernew);
    if (empty($USER->newadminuser) && isset($usernew->interests)) {
        useredit_update_interests($usernew, $usernew->interests);
    }
    if (empty($USER->newadminuser)) {
        core_user::update_picture($usernew, $filemanageroptions);
    }
    useredit_update_bounces($user, $usernew);
    useredit_update_trackforums($user, $usernew);
    profile_save_data($usernew);

    // Handle custom role assignment
    if (!empty($usernew->custom_user_role)) {
        $context = context_system::instance();
        role_unassign_all(array('userid' => $usernew->id, 'contextid' => $context->id));
        role_assign($usernew->custom_user_role, $usernew->id, $context->id);
    }

    // Reload from db.
    $usernew = $DB->get_record('user', array('id' => $usernew->id));

    if ($createpassword) {
        setnew_password_and_mail($usernew);
        unset_user_preference('create_password', $usernew);
        set_user_preference('auth_forcepasswordchange', 1, $usernew);
    }

    if ($usercreated) {
        \core\event\user_created::create_from_userid($usernew->id)->trigger();
    } else {
        \core\event\user_updated::create_from_userid($usernew->id)->trigger();
    }

    // If the edited user is the current user, refresh session $USER
    if ($user->id == $USER->id) {
        foreach ((array)$usernew as $variable => $value) {
            if ($variable === 'description' or $variable === 'password') {
                continue;
            }
            $USER->$variable = $value;
        }
        profile_load_custom_fields($USER);

        if (!empty($USER->newadminuser)) {
            unset($USER->newadminuser);
            admin_apply_default_settings(null, false);
            unset_config('adminsetuppending');
            redirect("$CFG->wwwroot/$CFG->admin/");
        } else if (empty($SITE->fullname)) {
            redirect("$CFG->wwwroot/$CFG->admin/");
        } else {
            redirect($returnurl, get_string('changessaved'), null, \core\output\notification::NOTIFY_SUCCESS);
        }
    } else if ($returnto === 'profile') {
        \core\session\manager::gc();
        redirect($returnurl, get_string('changessaved'), null, \core\output\notification::NOTIFY_SUCCESS);
    } else {
        \core\session\manager::gc();
        redirect("$CFG->wwwroot/$CFG->admin/user.php",'Created Successfully', null, \core\output\notification::NOTIFY_SUCCESS);
    }
}

// Display page header.
if ($user->id == -1 or ($user->id != $USER->id)) {
    if ($user->id == -1) {
        echo $OUTPUT->header();
    } else {
        $streditmyprofile = get_string('editmyprofile');
        $userfullname = fullname($user, true);
        $PAGE->set_heading($userfullname);
        $PAGE->set_title("$course->shortname: $streditmyprofile - $userfullname");
        echo $OUTPUT->header();
        echo $OUTPUT->heading($userfullname);
    }
} else if (!empty($USER->newadminuser)) {
    $strinstallation = get_string('installation', 'install');
    $strprimaryadminsetup = get_string('primaryadminsetup');
    $PAGE->navbar->add($strprimaryadminsetup);
    $PAGE->set_title($strinstallation);
    $PAGE->set_heading($strinstallation);
    $PAGE->set_cacheable(false);
    echo $OUTPUT->header();
    echo $OUTPUT->box(get_string('configintroadmin', 'admin'), 'generalbox boxwidthnormal boxaligncenter');
    echo '<br />';
} else {
    $streditmyprofile = get_string('editmyprofile');
    $userfullname     = fullname($user, true);
    $PAGE->set_title("$course->shortname: $streditmyprofile");
    $PAGE->set_heading($userfullname);
    echo $OUTPUT->header();
    echo $OUTPUT->heading($streditmyprofile);
}

// Finally display the form.
$userform->display();

// Hide secondary navigation for cleaner UI (you had this in your original)
echo '<script>
document.addEventListener("DOMContentLoaded", function() {
    var elements = document.querySelectorAll(".secondary-navigation.d-print-none");
    elements.forEach(function(el) {
        el.style.display = "none";
    });
});
</script>';

// Sync email -> username in the form UI
$PAGE->requires->js_init_code("
    document.addEventListener('DOMContentLoaded', function() {
        var emailField = document.querySelector('input[name=\"email\"]');
        var usernameField = document.querySelector('input[name=\"username\"]');

        if (emailField && usernameField) {
            emailField.addEventListener('input', function() {
                usernameField.value = emailField.value;
            });
            usernameField.value = emailField.value || usernameField.value;
        }
    });
");

echo $OUTPUT->footer();
