<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * This script displays a particular page of a quiz attempt that is in progress.
 *
 * @package   mod_quiz
 * @copyright 1999 onwards Martin Dougiamas  {@link http://moodle.com}
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(__DIR__ . '/../../config.php');
require_once($CFG->dirroot . '/mod/quiz/locallib.php');

// Look for old-style URLs, such as may be in the logs, and redirect them to startattemtp.php.
if ($id = optional_param('id', 0, PARAM_INT)) {
    redirect($CFG->wwwroot . '/mod/quiz/startattempt.php?cmid=' . $id . '&sesskey=' . sesskey());
} else if ($qid = optional_param('q', 0, PARAM_INT)) {
    if (!$cm = get_coursemodule_from_instance('quiz', $qid)) {
        print_error('invalidquizid', 'quiz');
    }
    redirect(new moodle_url('/mod/quiz/startattempt.php',
            array('cmid' => $cm->id, 'sesskey' => sesskey())));
}

// Get submitted parameters.
$attemptid = required_param('attempt', PARAM_INT);
$page = optional_param('page', 0, PARAM_INT);
$cmid = optional_param('cmid', null, PARAM_INT);

$attemptobj = quiz_create_attempt_handling_errors($attemptid, $cmid);
$page = $attemptobj->force_page_number_into_range($page);
$PAGE->set_url($attemptobj->attempt_url(null, $page));
// During quiz attempts, the browser back/forwards buttons should force a reload.
$PAGE->set_cacheable(false);

$PAGE->set_secondary_active_tab("modulepage");

// Check login.
require_login($attemptobj->get_course(), false, $attemptobj->get_cm());

// Check that this attempt belongs to this user.
if ($attemptobj->get_userid() != $USER->id) {
    if ($attemptobj->has_capability('mod/quiz:viewreports')) {
        redirect($attemptobj->review_url(null, $page));
    } else {
        throw new moodle_quiz_exception($attemptobj->get_quizobj(), 'notyourattempt');
    }
}

// Check capabilities and block settings.
if (!$attemptobj->is_preview_user()) {
    $attemptobj->require_capability('mod/quiz:attempt');
    if (empty($attemptobj->get_quiz()->showblocks)) {
        $PAGE->blocks->show_only_fake_blocks();
    }

} else {
    navigation_node::override_active_url($attemptobj->start_attempt_url());
}

// If the attempt is already closed, send them to the review page.
if ($attemptobj->is_finished()) {
    redirect($attemptobj->review_url(null, $page));
} else if ($attemptobj->get_state() == quiz_attempt::OVERDUE) {
    redirect($attemptobj->summary_url());
}

// Check the access rules.
$accessmanager = $attemptobj->get_access_manager(time());
$accessmanager->setup_attempt_page($PAGE);
$output = $PAGE->get_renderer('mod_quiz');
$messages = $accessmanager->prevent_access();
if (!$attemptobj->is_preview_user() && $messages) {
    print_error('attempterror', 'quiz', $attemptobj->view_url(),
            $output->access_messages($messages));
}
if ($accessmanager->is_preflight_check_required($attemptobj->get_attemptid())) {
    redirect($attemptobj->start_attempt_url(null, $page));
}

// Set up auto-save if required.
$autosaveperiod = get_config('quiz', 'autosaveperiod');
if ($autosaveperiod) {
    $PAGE->requires->yui_module('moodle-mod_quiz-autosave',
            'M.mod_quiz.autosave.init', array($autosaveperiod));
}

// Log this page view.
$attemptobj->fire_attempt_viewed_event();

// Get the list of questions needed by this page.
$slots = $attemptobj->get_slots($page);

// Check.
if (empty($slots)) {
    throw new moodle_quiz_exception($attemptobj->get_quizobj(), 'noquestionsfound');
}

// Update attempt page, redirecting the user if $page is not valid.
if (!$attemptobj->set_currentpage($page)) {
    redirect($attemptobj->start_attempt_url(null, $attemptobj->get_currentpage()));
}

// Initialise the JavaScript.
$headtags = $attemptobj->get_html_head_contributions($page);
$PAGE->requires->js_init_call('M.mod_quiz.init_attempt_form', null, false, quiz_get_js_module());
\core\session\manager::keepalive(); 


/* =====================================================
   PAGE SETUP
   ===================================================== */
$PAGE->set_title($attemptobj->attempt_page_title($page));
$PAGE->add_body_class('limitedwidth');
$PAGE->set_heading($attemptobj->get_course()->fullname);
$PAGE->activityheader->disable();

$nextpage = $attemptobj->is_last_page($page) ? -1 : $page + 1;

/* =====================================================
   CUSTOM QUESTION NAVIGATION DATA
   ===================================================== */

/* =====================================================
   QUESTION DATA
   ===================================================== */
$allslots = $attemptobj->get_slots();
$questionstates = [];
$flagged = [];

foreach ($allslots as $slot) {
    $qa = $attemptobj->get_question_attempt($slot);
    $questionstates[$slot] = $qa->get_state();
    $flagged[$slot] = $qa->is_flagged();
}

/* ===== CORRECT EXAM LOGIC ===== */
$currentpage = $attemptobj->get_currentpage();
$total = count($allslots);
$attempted = $skipped = $remaining = $review = 0;

$slotpage = [];
foreach ($allslots as $s) {
    $slotpage[$s] = $attemptobj->get_question_page($s);
}

foreach ($allslots as $slot) {
    if ($flagged[$slot]) {
        $review++;
    }

    $state = $questionstates[$slot];
    $qpage = $slotpage[$slot];

    if (in_array($state, [
        question_state::$complete,
        question_state::$gradedright,
        question_state::$gradedwrong,
        question_state::$gradedpartial
    ])) {
        $attempted++;
    } else if ($qpage < $currentpage) {
        $skipped++;
    } else {
        $remaining++;
    }
}

echo '<style>

/* ===== TIMER WRAPPER ===== */
#quiz-timer-wrapper{
  width:100%;
  display:flex;
  justify-content:flex-end;
  margin-bottom:10px;
}

/* ===== TIMER BOX ===== */
#quiz-timer{
  background:#ffffff;
  border:1px solid #e0e0e0;
  border-radius:6px;
  padding:8px 12px;
  min-width:160px;
  text-align:center;
  font-weight:700;
  font-size:16px;
  position:relative;
}

/* ICON */
#quiz-timer .timer-icon{
  margin-right:6px;
}

/* ===== PROGRESS BAR ===== */
.timer-bar{
  width:100%;
  height:6px;
  background:#e9ecef;
  border-radius:4px;
  margin-top:6px;
  overflow:hidden;
}

.timer-progress{
  height:100%;
  width:100%;
  background:linear-gradient(90deg,#28a745,#ffc107,#dc3545);
  transition:width 1s linear;
}

</style>';

/* =====================================================
   CSS (ATTRACTIVE EXAM STYLE)
   ===================================================== */

echo '<style>

/* ===== LAYOUT WRAPPER (NEW – SAFE) ===== */
.exam-layout{
  display:grid;
  grid-template-columns:260px 1fr;
  gap:24px;
  align-items:flex-start;
}

@media(max-width:768px){
  .exam-layout{
    grid-template-columns:1fr;
  }
}

/* ===== CENTER LAYOUT ===== */
.exam-center{
  width:100%;
  display:flex;
  flex-direction:column;
  align-items:center;
  margin:20px 0;
}

/* ===== QUESTION PALETTE ===== */
.question-palette{
  display:grid;
  grid-template-columns:repeat(auto-fit, 36px);
  gap:10px;
  justify-content:center;
  max-width:560px;
  width:100%;
}

.qbox{
  width:36px;
  height:36px;
  display:flex;
  align-items:center;
  justify-content:center;
  font-size:14px;
  font-weight:600;
  text-decoration:none;
  border-radius:4px;
  background:#e9ecef;
  color:#212529;
}

.qbox.attempted{background:#d1e7dd;color:#0f5132}
.qbox.skipped{background:#f8d7da;color:#842029}
.qbox.current{background:#0d6efd;color:#fff}
.qbox.remaining{background:#e9ecef}
.qbox.review{background:#fff3cd;border:1px solid #ffca2c}
.review .count{background:#fd7e14}

/* ===== STATUS FILTERS ===== */
.exam-status{
  display:flex;
  gap:10px;
  margin-top:16px;
  max-width:560px;
  width:100%;
}

.exam-status button{
  flex:1;
  padding:8px 0;
  font-size:13px;
  font-weight:600;
  border-radius:4px;
  border:1px solid #ced4da;
  background:#f8f9fa;
  cursor:pointer;
}

.exam-status button.active{
  background:#0d6efd;
  color:#fff;
  border-color:#0d6efd;
}

/* ❌ NO HOVER EFFECT */
.exam-status button:hover{
  background:#f8f9fa;
  color:inherit;
}

/* ===== MOBILE ===== */
@media(max-width:480px){
  .question-palette{
    gap:8px;
  }
}
</style>';


/* ===== QUESTION SUMMARY (UNCHANGED) ===== */
echo '<style>
/* ===== QUESTION SUMMARY ===== */
.exam-summary{
  border:1px solid #dee2e6;
  border-radius:8px;
  background:#ffffff;
  padding:14px;
  max-width:260px;
  width:100%;
  box-shadow:0 2px 6px rgba(0,0,0,0.05);
}

/* Title */
.exam-summary h4{
  font-size:15px;
  font-weight:600;
  margin:0 0 12px;
  padding-bottom:6px;
  border-bottom:1px solid #e9ecef;
  text-align:center;
}

/* Summary rows */
.exam-summary .summary-item{
  display:flex;
  justify-content:space-between;
  align-items:center;
  padding:6px 0;
  font-size:14px;
}

/* Labels */
.exam-summary .label{
  color:#495057;
}

/* Count badge */
.exam-summary .count{
  min-width:28px;
  text-align:center;
  padding:2px 8px;
  border-radius:12px;
  font-weight:600;
  font-size:13px;
}

/* Colors */
.exam-summary .total .count{
  background:#0d6efd;
  color:#fff;
}
.exam-summary .attempted .count{
  background:#198754;
  color:#fff;
}
.exam-summary .skipped .count{
  background:#dc3545;
  color:#fff;
}
.exam-summary .remaining .count{
  background:#6c757d;
  color:#fff;
}
</style>';




$customnav = '
<div class="exam-layout">

  <!-- QUESTION SUMMARY -->
  <div class="exam-summary">
    <h4>Question Summary</h4>

    <div class="summary-item total">
      <span class="label">Total Questions</span>
      <span class="count">'.$total.'</span>
    </div>

    <div class="summary-item attempted">
      <span class="label">Attempted</span>
      <span class="count">'.$attempted.'</span>
    </div>

    <div class="summary-item skipped">
      <span class="label">Skipped</span>
      <span class="count">'.$skipped.'</span>
    </div>

    <div class="summary-item remaining">
      <span class="label">Remaining</span>
      <span class="count">'.$remaining.'</span>
    </div>

 <div class="summary-item review"><span>Marked for Review</span><span class="count">'.$review.'</span></div>
</div>



  <!-- RIGHT : PALETTE + STATUS -->
  <div class="exam-center">

    <div class="question-palette">';


foreach ($allslots as $slot) {

    $state = $questionstates[$slot];
    $status = "remaining";

    if (in_array($state, [
        question_state::$complete,
        question_state::$gradedright,
        question_state::$gradedwrong,
        question_state::$gradedpartial
    ])) {
        $status = "attempted";
    } else if ($slotpage[$slot] < $currentpage) {
        $status = "skipped";
    }

    if (in_array($slot, $slots)) {
        $status = "current";
    }

    $url = new moodle_url("/mod/quiz/attempt.php", [
        "attempt" => $attemptid,
        "page" => $attemptobj->get_question_page($slot)
    ]);

  $customnav .= '<a href="'.$url.'" class="qbox '.$status.'" data-status="'.$status.'">'.$slot.'</a>';


}

$customnav .= '
  </div>

  <div class="exam-status">
   <button data-filter="all" class="active" style="color:#000;">All</button>
<button data-filter="current" style="color:#000;">Current</button>
<button data-filter="attempted" style="color:#000;">Attempted</button>
<button data-filter="skipped" style="color:#000;">Skipped</button>

  </div>

</div>';


/* =====================================================
   JS FILTER
   ===================================================== */
echo '<script>
document.addEventListener("DOMContentLoaded", function () {

  document.body.addEventListener("click", function (e) {

    const btn = e.target.closest(".exam-status button");
    if (!btn) return;

    // Active class
    document.querySelectorAll(".exam-status button")
      .forEach(b => b.classList.remove("active"));
    btn.classList.add("active");

    const filter = btn.getAttribute("data-filter");

    document.querySelectorAll(".question-palette .qbox").forEach(q => {

      // ALL
      if (filter === "all") {
        q.style.display = "inline-flex";
        return;
      }

      // CURRENT
      if (filter === "current") {
        q.style.display = q.classList.contains("current")
          ? "inline-flex"
          : "none";
        return;
      }

      // ATTEMPTED / SKIPPED / REMAINING
      if (q.classList.contains(filter)) {
        q.style.display = "inline-flex";
      } else {
        q.style.display = "none";
      }
    });
  });

});
</script>';

echo '
<script>
document.addEventListener("DOMContentLoaded", function () {
    var nextBtn = document.querySelector(
        "input#mod_quiz-next-nav.mod_quiz-next-nav[name=next]"
    );
    if (nextBtn) {
        nextBtn.value = "Save & Next";
    }
});
</script>
';// Try to prevent sessions expiring during quiz attempts.

// Arrange for the navigation to be displayed in the first region on the page.
$navbc = $attemptobj->get_navigation_panel($output, 'quiz_attempt_nav_panel', $page);
$regions = $PAGE->blocks->get_regions();
$PAGE->blocks->add_fake_block($navbc, reset($regions));

$headtags = $attemptobj->get_html_head_contributions($page);
$PAGE->set_title($attemptobj->attempt_page_title($page));
$PAGE->add_body_class('limitedwidth');
$PAGE->set_heading($attemptobj->get_course()->fullname);
$PAGE->activityheader->disable();
if ($attemptobj->is_last_page($page)) {
    $nextpage = -1;
} else {
    $nextpage = $page + 1;
}
echo '<style>
.secondary-navigation.d-print-none {
    display: none !important;
}


.editquestion{
     display: none !important;
}

</style>';




/* =====================================================
   RENDER PAGE & INSERT AFTER region-main
   ===================================================== */
ob_start();
echo $output->attempt_page(
    $attemptobj,
    $page,
    $accessmanager,
    $messages,
    $slots,
    $id ?? null,
    $nextpage
);
$pagehtml = ob_get_clean();

/*
 * Insert custom navigation JUST AFTER:
 * <section id="region-main" aria-label="Content">
 */
$pagehtml = preg_replace(
    '/(<footer[^>]*id=["\']page-footer["\'][^>]*>)/i',
    $customnav . '$1',
    $pagehtml,
    1
);

echo $pagehtml;
//echo $output->attempt_page($attemptobj, $page, $accessmanager, $messages, $slots, $id, $nextpage);
