<?php
require_once(__DIR__ . '/../config.php');
require_login();

// Page setup
$PAGE->set_url(new moodle_url('/course/wcourse_domain.php'));
$PAGE->set_context(context_system::instance());
$PAGE->set_title('Courses by Domain / Language');
$PAGE->set_heading('Courses by Domain / Language');

// Pagination
$perpage = 10;
$page = optional_param('page', 0, PARAM_INT);
$offset = $page * $perpage;

// Filters
$selecteddomain = optional_param('domain', '', PARAM_TEXT);
$selectedlanguage = optional_param('language', '', PARAM_TEXT);
$search = optional_param('search', '', PARAM_RAW_TRIMMED);

/* --------------------------
   1️⃣ GET DOMAIN CUSTOM FIELD
-------------------------- */
$domainfield = $DB->get_record('customfield_field', ['shortname' => 'domain'], '*', IGNORE_MISSING);
if (!$domainfield) {
    echo $OUTPUT->header();
    echo $OUTPUT->notification('Custom field "domain" not found in system.', 'error');
    echo $OUTPUT->footer();
    exit;
}
$fieldid = $domainfield->id;

/* --------------------------
   2️⃣ LOAD DOMAIN LIST
-------------------------- */
$domains = $DB->get_records_sql_menu("
    SELECT DISTINCT TRIM(value) AS value, TRIM(value) AS displayname
    FROM {customfield_data}
    WHERE fieldid = :fieldid AND value <> ''
    ORDER BY value ASC", ['fieldid' => $fieldid]);


/* --------------------------
   3️⃣ LANGUAGE NAME MAPPING
-------------------------- */
$lang_names = [
    'ar-SA' => 'Arabic (Saudi Arabia)',
    'de-DE' => 'German (Germany)',
    'en-US' => 'English (United States)',
    'es-419' => 'Spanish (Latin America)',
    'es-ES' => 'Spanish (Spain)',
    'fr-FR' => 'French (France)',
    'id-ID' => 'Indonesian',
    'it-IT' => 'Italian',
    'ja-JP' => 'Japanese',
    'ko-KR' => 'Korean',
    'pt-BR' => 'Portuguese (Brazil)',
    'zh-CN' => 'Chinese (Simplified)',
    'zh-TW' => 'Chinese (Traditional)',
    'th-TH' => 'Thai',
    'vi-VN' => 'Vietnamese',
    'he-IL' => 'Hebrew',
    'el-GR' => 'Greek',
    'pl-PL' => 'Polish',
    'tr-TR' => 'Turkish',
    'uk-UA' => 'Ukrainian'
];

/* --------------------------
   4️⃣ LOAD LANGUAGES FROM COURSES
-------------------------- */
$rawlangs = $DB->get_records_sql_menu("
    SELECT DISTINCT lang AS id, lang
    FROM {course}
    WHERE lang IS NOT NULL AND lang <> ''
    ORDER BY lang ASC
");

$languageoptions = [];
foreach ($rawlangs as $code => $val) {
    $languageoptions[$code] = $lang_names[$code] ?? strtoupper($code);
}

/* --------------------------
   5️⃣ BUILD DYNAMIC SQL
-------------------------- */
$params = ['fieldid' => $fieldid];
$where = ["d.fieldid = :fieldid"];

if (!empty($selecteddomain)) {
    $where[] = "d.value = :selecteddomain";
    $params['selecteddomain'] = $selecteddomain;
}

if (!empty($selectedlanguage)) {
    $where[] = "c.lang = :selectedlanguage";
    $params['selectedlanguage'] = $selectedlanguage;
}

if (!empty($search)) {
    $where[] = $DB->sql_like('c.fullname', ':search', false, false);
    $params['search'] = '%' . $search . '%';
}

$whereclause = implode(" AND ", $where);

$sql = "SELECT c.id, c.fullname
        FROM {course} c
        JOIN {customfield_data} d ON d.instanceid = c.id
        WHERE $whereclause
        ORDER BY c.fullname ASC";

// Count total
$totalcourses = $DB->count_records_sql("SELECT COUNT(1) FROM ($sql) temp", $params);

// Apply pagination
$sql .= " LIMIT $perpage OFFSET $offset";
$courses = $DB->get_records_sql($sql, $params);

/* --------------------------
   6️⃣ OUTPUT START
-------------------------- */
echo $OUTPUT->header();
echo html_writer::tag('h3', 'Courses', ['class' => 'mb-3']);

/* --------------------------
   7️⃣ FILTER BAR FORM
-------------------------- */
echo html_writer::start_tag('form', ['method' => 'get', 'action' => $PAGE->url]);
echo html_writer::start_div('filter-bar', [
    'style' => 'display:flex;gap:10px;align-items:center;margin-bottom:15px;flex-wrap:wrap;'
]);

/* --- Domain Dropdown --- */
echo html_writer::start_tag('select', [
    'name' => 'domain',
    'onchange' => 'this.form.submit()',
    'style' => 'padding:5px;'
]);
echo html_writer::tag('option', 'All Domains', ['value' => '']);
foreach ($domains as $domainvalue => $domainname) {
    $selected = ($domainvalue === $selecteddomain) ? ['selected' => 'selected'] : [];
    echo html_writer::tag('option', $domainname, ['value' => $domainvalue] + $selected);
}
echo html_writer::end_tag('select');

/* --- Language Dropdown --- */
echo html_writer::start_tag('select', [
    'name' => 'language',
    'onchange' => 'this.form.submit()',
    'style' => 'padding:5px;'
]);
echo html_writer::tag('option', 'All Languages', ['value' => '']);
foreach ($languageoptions as $code => $name) {
    $selected = ($code === $selectedlanguage) ? ['selected' => 'selected'] : [];
    echo html_writer::tag('option', $name, ['value' => $code] + $selected);
}
echo html_writer::end_tag('select');

/* --- Search Box --- */
echo html_writer::empty_tag('input', [
    'type' => 'text',
    'name' => 'search',
    'value' => $search,
    'placeholder' => 'Search course name...',
    'style' => 'padding:5px;width:250px;'
]);

/* --- Search Button --- */
echo html_writer::empty_tag('input', [
    'type' => 'submit',
    'value' => 'Search',
    'style' => 'padding:5px 10px;cursor:pointer;'
]);

echo html_writer::end_div();
echo html_writer::end_tag('form');

/* --------------------------
   8️⃣ COURSE LIST
-------------------------- */
if ($courses) {
    echo html_writer::start_tag('ul', ['style' => 'list-style:none;padding:0;']);
    foreach ($courses as $course) {
        //$courseurl = new moodle_url('/course/view.php', ['id' => $course->id]);
        echo html_writer::tag('li',
            html_writer::link($courseurl, format_string($course->fullname), [
                'style' => 'display:block;padding:10px 15px;margin-bottom:8px;border:1px solid #ddd;
                            border-radius:6px;text-decoration:none;color:#FF4B00;font-weight:500;'
            ])
        );
    }
    echo html_writer::end_tag('ul');
} else {
    echo html_writer::div('No courses found for selected filters.', 'alert alert-info');
}

/* --------------------------
   9️⃣ PAGINATION
-------------------------- */
$baseurl = new moodle_url('/course/wcourse_domain.php', [
    'domain' => $selecteddomain,
    'language' => $selectedlanguage,
    'search' => $search
]);
echo $OUTPUT->paging_bar($totalcourses, $page, $perpage, $baseurl);

// Footer
echo $OUTPUT->footer();
?>
