<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Form for editing a users profile (customized).
 *
 * @package core_user
 */

defined('MOODLE_INTERNAL') || die('Direct access to this script is forbidden.');

require_once($CFG->dirroot.'/lib/formslib.php');
require_once($CFG->dirroot.'/user/lib.php');

class user_editadvanced_form extends moodleform {

    /**
     * Define the form.
     */
    public function definition() {
        global $USER, $CFG, $COURSE, $DB;

        $mform = $this->_form;

        if (!is_array($this->_customdata)) {
            throw new coding_exception('invalid custom data for user_edit_form');
        }
        $editoroptions = $this->_customdata['editoroptions'];
        $filemanageroptions = $this->_customdata['filemanageroptions'];
        $user = $this->_customdata['user'];
        $userid = isset($user->id) ? $user->id : 0;

        // Accessibility: "Required" is bad legend text.
        $strgeneral  = get_string('general');
        $strrequired = get_string('required');

        // Hidden fields
        $mform->addElement('hidden', 'id');
        $mform->setType('id', core_user::get_property_type('id'));
        $mform->addElement('hidden', 'course', $COURSE->id);
        $mform->setType('course', PARAM_INT);

        // Header
        $mform->addElement('header', 'moodle', $strgeneral);

        // Username hidden and set default from email if available.
        $mform->addElement('hidden', 'username');
        $mform->setType('username', PARAM_RAW);
        if (!empty($user->email)) {
            $mform->setDefault('username', $user->email);
        }

        // Email, firstname, lastname come from useredit_shared_definition (called later)

        // Roles handling
        $allroles = role_get_names(null, ROLENAME_BOTH);
        $roleoptions = array();

        $editinguserid = !empty($this->_customdata['user']->id) ? $this->_customdata['user']->id : 0;
        $is_own_profile = ($USER->id == $editinguserid);

        $context = context_system::instance();
        $userroles = get_user_roles($context, $USER->id);
        $currentuserrole = !empty($userroles) ? reset($userroles) : null;
        $currentuserrolename = $currentuserrole ? strtolower($currentuserrole->shortname) : '';

        if (!($is_own_profile && ($USER->id == 1 || $USER->id == 2 || $currentuserrolename === 'manager'))) {
            if ($USER->id == 1 || $USER->id == 2) {
                foreach ($allroles as $roleid => $role) {
                    if (strtolower($role->shortname) === 'manager') {
                        $roleoptions[$roleid] = $role->localname;
                    }
                }
            } else {
                foreach ($allroles as $roleid => $role) {
                    if (strtolower($role->shortname) === 'student') {
                        $roleoptions[$roleid] = $role->localname;
                    }
                }
            }

            if (!empty($roleoptions)) {
                $mform->addElement('select', 'custom_user_role', get_string('roles'), $roleoptions);
                $mform->setType('custom_user_role', PARAM_INT);

                // default role when editing existing user
                if (!empty($this->_customdata['user']->id) && $this->_customdata['user']->id > 0) {
                    $assignedroles = get_user_roles($context, $this->_customdata['user']->id);
                    if (!empty($assignedroles)) {
                        $role = reset($assignedroles);
                        $mform->setDefault('custom_user_role', $role->roleid);
                    }
                }
            }
        }

        // Employee ID field
        $mform->addElement('text', 'emp_id', 'Employee ID', 'size="20"');
        $mform->setType('emp_id', PARAM_TEXT);
        if (!empty($user->emp_id)) {
            $mform->setDefault('emp_id', $user->emp_id);
        }

        // Unique Employee ID validation (simple)
        $mform->addRule('emp_id', 'Employee ID must be unique', 'callback', function($value) use ($DB) {
            if (trim($value) === '') {
                return true;
            }
            $editinguserid = optional_param('id', 0, PARAM_INT);
            $exists = $DB->record_exists_select('user', 'emp_id = :emp AND id != :id', [
                'emp' => $value,
                'id'  => $editinguserid
            ]);
            return !$exists;
        });

        // Company dropdown (load from 'company' table)
        $companies = [];
        if ($DB->get_manager()->table_exists('company')) {
            $companies = $DB->get_records_menu('company', [], 'com_name ASC', 'id, com_name');
        }
        // Add a default empty option
        $companyoptions = array(0 => get_string('choose')) + (array)$companies;
        $mform->addElement('select', 'companyid', 'Company', $companyoptions);
        $mform->setType('companyid', PARAM_INT);

        // If editing, set default company. We support two fields: companyid (form) and company_id (DB column)
        $existingcompany = 0;
        if (!empty($user->companyid)) {
            $existingcompany = (int)$user->companyid;
        } else if (!empty($user->company_id)) {
            $existingcompany = (int)$user->company_id;
        }
        if ($existingcompany) {
            $mform->setDefault('companyid', $existingcompany);
        }

        // Make company required (if you want)
        $mform->addRule('companyid', get_string('required'), 'required', null, 'client');

        // Password field behavior (existing code preserved)
        $purpose = user_edit_map_field_purpose($userid, 'password');
        if ($userid < 1 && isset($user->auth) && $user->auth === 'manual') {
            $mform->addElement('hidden', 'newpassword', '');
            $mform->setType('newpassword', PARAM_RAW);
            $mform->addElement('hidden', 'newpassword2', '');
            $mform->setType('newpassword2', PARAM_RAW);
            require_once($CFG->libdir . '/weblib.php');
            $randompassword = generate_password();
            $this->_form->setDefault('newpassword', $randompassword);
            $this->_form->setDefault('newpassword2', $randompassword);
        } else {
            $mform->addElement('passwordunmask', 'newpassword', get_string('newpassword'), 'size="20" ' . $purpose);
            $mform->addHelpButton('newpassword', 'newpassword');
            $mform->setType('newpassword', core_user::get_property_type('password'));
        }

        // Shared fields and profile fields (use core functions)
        useredit_shared_definition($mform, $editoroptions, $filemanageroptions, $user);
        profile_definition($mform, $userid);

        // Buttons
        if ($userid == -1) {
            $btnstring = get_string('createuser');
        } else {
            $btnstring = get_string('updatemyprofile');
        }
        $this->add_action_buttons(true, $btnstring);

        // JS to hide specific elements
        $mform->addElement('html', '
<script>
document.addEventListener("DOMContentLoaded", function() {
    [
        "id_moodle_picture",
        "id_moodle_additional_names",
        "id_moodle_interests"
    ].forEach(function(id) {
        const el = document.getElementById(id);
        if (el) {
            el.style.display = "none";
        }
    });
});
</script>
');
        // Map DB field to form field so dropdown gets selected in edit mode
if (!empty($user->company_id)) {
    $user->companyid = (int)$user->company_id;
}


        // Populate form with user data passed in
        $this->set_data($user);
    }

    /**
     * Extend the form definition after data has been parsed.
     */
    public function definition_after_data() {
        global $USER, $CFG, $DB, $OUTPUT;

        $mform = $this->_form;

        // Trim required name fields.
        foreach (useredit_get_required_name_fields() as $field) {
            $mform->applyFilter($field, 'trim');
        }

        if ($userid = $mform->getElementValue('id')) {
            $user = $DB->get_record('user', array('id' => $userid));
        } else {
            $user = false;
        }

        if ($userid == $USER->id) {
            if ($mform->elementExists('auth')) {
                $mform->hardFreeze('auth');
            }
            if ($mform->elementExists('preference_auth_forcepasswordchange')) {
                $mform->hardFreeze('preference_auth_forcepasswordchange');
            }
        }

        if ($user and is_mnet_remote_user($user)) {
            if ($mform->elementExists('suspended')) {
                $mform->removeElement('suspended');
            }
        }
        if ($user and ($user->id == $USER->id or is_siteadmin($user))) {
            if ($mform->elementExists('suspended')) {
                $mform->hardFreeze('suspended');
            }
        }

        if (empty($USER->newadminuser)) {
            if ($user) {
                $context = context_user::instance($user->id, MUST_EXIST);
                $fs = get_file_storage();
                $hasuploadedpicture = ($fs->file_exists($context->id, 'user', 'icon', 0, '/', 'f2.png') || $fs->file_exists($context->id, 'user', 'icon', 0, '/', 'f2.jpg'));
                if (!empty($user->picture) && $hasuploadedpicture) {
                    $imagevalue = $OUTPUT->user_picture($user, array('courseid' => SITEID, 'size' => 64));
                } else {
                    $imagevalue = get_string('none');
                }
            } else {
                $imagevalue = get_string('none');
            }
            if ($mform->elementExists('currentpicture')) {
                $imageelement = $mform->getElement('currentpicture');
                if ($imageelement) {
                    $imageelement->setValue($imagevalue);
                }
                if ($user && $mform->elementExists('deletepicture') && !$hasuploadedpicture) {
                    $mform->removeElement('deletepicture');
                }
            }
        }

        profile_definition_after_data($mform, $userid);
    }

    /**
     * Validate the form data.
     * @param array $usernew
     * @param array $files
     * @return array|bool
     */
    public function validation($usernew, $files) {
        global $CFG, $DB;

        $usernew = (object)$usernew;
        $usernew->username = trim($usernew->username);

        $user = null;
        if (!empty($usernew->id)) {
            $user = $DB->get_record('user', array('id' => $usernew->id));
        }

        $err = array();

        if (!$user && !empty($usernew->createpassword)) {
            if (!empty($usernew->suspended)) {
                $err['suspended'] = get_string('error');
            }
        } else {
            if (!empty($usernew->newpassword)) {
                $errmsg = '';
                if (!check_password_policy($usernew->newpassword, $errmsg, $usernew)) {
                    $err['newpassword'] = $errmsg;
                }
            } else if (!$user) {
                $auth = get_auth_plugin($usernew->auth);
                if ($auth->is_internal() && $usernew->auth !== 'manual') {
                    $err['newpassword'] = get_string('required');
                }
            }
        }

        if (empty($usernew->username)) {
            $err['username'] = get_string('required');
        } else if (!$user or $user->username !== $usernew->username) {
            if ($DB->record_exists('user', array('username' => $usernew->username, 'mnethostid' => $CFG->mnet_localhost_id))) {
                $err['username'] = get_string('usernameexists');
            }
            if ($usernew->username !== core_text::strtolower($usernew->username)) {
                $err['username'] = get_string('usernamelowercase');
            } else {
                if ($usernew->username !== core_user::clean_field($usernew->username, 'username')) {
                    $err['username'] = get_string('invalidusername');
                }
            }
        }

        if (!$user or (isset($usernew->email) && $user->email !== $usernew->email)) {
            if (!validate_email($usernew->email)) {
                $err['email'] = get_string('invalidemail');
            } else if (empty($CFG->allowaccountssameemail)) {
                $select = $DB->sql_equal('email', ':email', false) . ' AND mnethostid = :mnethostid AND id <> :userid';
                $params = array(
                    'email' => $usernew->email,
                    'mnethostid' => $CFG->mnet_localhost_id,
                    'userid' => $usernew->id
                );
                if ($DB->record_exists_select('user', $select, $params)) {
                    $err['email'] = get_string('emailexists');
                }
            }
        }

        $err += profile_validation($usernew, $files);

        if (count($err) == 0) {
            return true;
        } else {
            return $err;
        }
    }
}
